/*
 * Decompiled with CFR 0.152.
 */
package com.android.utils;

import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Iterables;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;

public final class FileUtils {
    private static final Joiner PATH_JOINER = Joiner.on(File.separatorChar);
    private static final Joiner COMMA_SEPARATED_JOINER = Joiner.on(", ");
    private static final Joiner UNIX_NEW_LINE_JOINER = Joiner.on('\n');

    private FileUtils() {
    }

    public static void deletePath(File path) throws IOException {
        if (!path.exists()) {
            return;
        }
        if (path.isDirectory()) {
            FileUtils.deleteDirectoryContents(path);
        }
        if (!path.delete()) {
            throw new IOException(String.format("Could not delete path '%s'.", path));
        }
    }

    public static void deleteDirectoryContents(File directory) throws IOException {
        Preconditions.checkArgument(directory.isDirectory(), "!directory.isDirectory");
        File[] files = directory.listFiles();
        Preconditions.checkNotNull(files);
        for (File file : files) {
            FileUtils.deletePath(file);
        }
    }

    public static void cleanOutputDir(File path) throws IOException {
        if (!path.isDirectory()) {
            if (path.exists()) {
                FileUtils.deletePath(path);
            }
            if (!path.mkdirs()) {
                throw new IOException(String.format("Could not create empty folder %s", path));
            }
            return;
        }
        FileUtils.deleteDirectoryContents(path);
    }

    public static void copyFile(File from, File to) throws IOException {
        Files.copy(from.toPath(), to.toPath(), StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING);
    }

    public static void copyDirectory(File from, File to) throws IOException {
        Preconditions.checkArgument(from.isDirectory(), "Source path is not a directory.");
        Preconditions.checkArgument(!to.exists() || to.isDirectory(), "Destination path exists and is not a directory.");
        FileUtils.mkdirs(to);
        File[] children = from.listFiles();
        if (children != null) {
            for (File child : children) {
                if (child.isFile()) {
                    FileUtils.copyFileToDirectory(child, to);
                    continue;
                }
                if (child.isDirectory()) {
                    FileUtils.copyDirectoryToDirectory(child, to);
                    continue;
                }
                throw new IllegalArgumentException("Don't know how to copy file " + child.getAbsolutePath());
            }
        }
    }

    public static void copyFileToDirectory(File from, File to) throws IOException {
        FileUtils.copyFile(from, new File(to, from.getName()));
    }

    public static void copyDirectoryToDirectory(File from, File to) throws IOException {
        FileUtils.copyDirectory(from, new File(to, from.getName()));
    }

    public static void copyDirectoryContentToDirectory(File from, File to) throws IOException {
        Preconditions.checkArgument(from.isDirectory(), "Source path is not a directory.");
        File[] children = from.listFiles();
        if (children != null) {
            for (File f : children) {
                File destination;
                if (f.isDirectory()) {
                    destination = new File(to, FileUtils.relativePath(f, from));
                    com.google.common.io.Files.createParentDirs(destination);
                    FileUtils.mkdirs(destination);
                    FileUtils.copyDirectoryContentToDirectory(f, destination);
                    continue;
                }
                if (!f.isFile()) continue;
                destination = new File(to, FileUtils.relativePath(f.getParentFile(), from));
                com.google.common.io.Files.createParentDirs(destination);
                FileUtils.mkdirs(destination);
                FileUtils.copyFileToDirectory(f, destination);
            }
        }
    }

    public static File mkdirs(File folder) {
        if (!folder.mkdirs() && !folder.isDirectory()) {
            throw new RuntimeException("Cannot create directory " + folder);
        }
        return folder;
    }

    public static void delete(File file) throws IOException {
        boolean result = file.delete();
        if (!result) {
            throw new IOException("Failed to delete " + file.getAbsolutePath());
        }
    }

    public static void deleteIfExists(File file) throws IOException {
        boolean result = file.delete();
        if (!result && file.exists()) {
            throw new IOException("Failed to delete " + file.getAbsolutePath());
        }
    }

    public static void renameTo(File file, File to) throws IOException {
        boolean result = file.renameTo(to);
        if (!result) {
            throw new IOException("Failed to rename " + file.getAbsolutePath() + " to " + to);
        }
    }

    public static File join(File dir, String ... paths) {
        if (paths.length == 0) {
            return dir;
        }
        return new File(dir, PATH_JOINER.join(paths));
    }

    public static File join(File dir, Iterable<String> paths) {
        return new File(dir, PATH_JOINER.join(paths));
    }

    public static String join(String ... paths) {
        return PATH_JOINER.join(paths);
    }

    public static String join(Iterable<String> paths) {
        return PATH_JOINER.join(paths);
    }

    public static String loadFileWithUnixLineSeparators(File file) throws IOException {
        return UNIX_NEW_LINE_JOINER.join(com.google.common.io.Files.readLines(file, Charsets.UTF_8));
    }

    public static String relativePath(File file, File dir) {
        Preconditions.checkArgument(file.isFile() || file.isDirectory(), "%s is not a file nor a directory.", (Object)file.getPath());
        Preconditions.checkArgument(dir.isDirectory(), "%s is not a directory.", (Object)dir.getPath());
        return FileUtils.relativePossiblyNonExistingPath(file, dir);
    }

    public static String relativePossiblyNonExistingPath(File file, File dir) {
        String path = dir.toURI().relativize(file.toURI()).getPath();
        return FileUtils.toSystemDependentPath(path);
    }

    public static String toSystemDependentPath(String path) {
        if (File.separatorChar != '/') {
            path = path.replace('/', File.separatorChar);
        }
        return path;
    }

    public static String toSystemIndependentPath(String path) {
        if (File.separatorChar != '/') {
            path = path.replace(File.separatorChar, '/');
        }
        return path;
    }

    public static String toExportableSystemDependentPath(File file) {
        if (File.separatorChar != '/' && !file.getAbsolutePath().startsWith("\\\\?\\")) {
            return "\\\\?\\" + file.getAbsolutePath();
        }
        return file.getAbsolutePath();
    }

    public static String sha1(File file) throws IOException {
        return Hashing.sha1().hashBytes(com.google.common.io.Files.toByteArray(file)).toString();
    }

    public static FluentIterable<File> getAllFiles(File dir) {
        return com.google.common.io.Files.fileTreeTraverser().preOrderTraversal(dir).filter(com.google.common.io.Files.isFile());
    }

    public static String getNamesAsCommaSeparatedList(Iterable<File> files) {
        return COMMA_SEPARATED_JOINER.join(Iterables.transform(files, File::getName));
    }

    public static String getDirectoryNameForJar(File inputFile) {
        HashFunction hashFunction = Hashing.sha1();
        HashCode hashCode = hashFunction.hashString(inputFile.getAbsolutePath(), Charsets.UTF_16LE);
        String name = com.google.common.io.Files.getNameWithoutExtension(inputFile.getName());
        if (name.equals("classes") && inputFile.getAbsolutePath().contains("exploded-aar")) {
            File versionDir = inputFile.getParentFile().getParentFile();
            File artifactDir = versionDir.getParentFile();
            File groupDir = artifactDir.getParentFile();
            name = Joiner.on('-').join(groupDir.getName(), artifactDir.getName(), versionDir.getName());
        }
        name = name + "_" + hashCode.toString();
        return name;
    }

    public static void createFile(File file, String content) throws IOException {
        Preconditions.checkArgument(!file.exists(), "%s exists already.", (Object)file);
        com.google.common.io.Files.createParentDirs(file);
        com.google.common.io.Files.write(content, file, Charsets.UTF_8);
    }

    public static List<File> find(File base, Pattern pattern) {
        Preconditions.checkArgument(base.isDirectory(), "'%s' must be a directory.", (Object)base.getAbsolutePath());
        return com.google.common.io.Files.fileTreeTraverser().preOrderTraversal(base).filter(file -> pattern.matcher(FileUtils.toSystemIndependentPath(file.getPath())).find()).toList();
    }

    public static Optional<File> find(File base, String name) {
        Preconditions.checkArgument(base.isDirectory(), "'%s' must be a directory.", (Object)base.getAbsolutePath());
        return com.google.common.io.Files.fileTreeTraverser().preOrderTraversal(base).filter(file -> name.equals(file.getName())).last();
    }

    public static String joinFilePaths(Iterable<File> files) {
        return Joiner.on(File.pathSeparatorChar).join(Iterables.transform(files, File::getAbsolutePath));
    }

    public static boolean parentDirExists(File file) {
        File canonicalFile;
        try {
            canonicalFile = file.getCanonicalFile();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return canonicalFile.getParentFile() != null && canonicalFile.getParentFile().exists();
    }

    public static boolean isFileInDirectory(File file, File directory) {
        try {
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        for (File parentFile = file.getCanonicalFile().getParentFile(); parentFile != null; parentFile = parentFile.getParentFile()) {
            if (!FileUtils.isSameFile(parentFile, directory)) continue;
            return true;
        }
        return false;
    }

    public static boolean isSameFile(File file1, File file2) {
        try {
            if (file1.exists() && file2.exists()) {
                return Files.isSameFile(file1.toPath(), file2.toPath());
            }
            return file1.getCanonicalFile().equals(file2.getCanonicalFile());
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static FileSystem createZipFilesystem(Path archive) throws IOException {
        URI uri = URI.create("jar:" + archive.toUri().toString());
        return FileSystems.newFileSystem(uri, Collections.emptyMap());
    }
}

