/*
 * $Header: /cvshome/build/org.osgi.service.deploymentadmin/src/org/osgi/service/deploymentadmin/spi/ResourceProcessorException.java,v 1.1 2005/11/03 12:42:21 izahoransky Exp $
 * 
 * Copyright (c) OSGi Alliance (2005). All Rights Reserved.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this 
 * distribution, and is available at http://www.eclipse.org/legal/epl-v10.html.
 */
package org.osgi.service.deploymentadmin.spi;

import org.osgi.service.deploymentadmin.spi.ResourceProcessor;

/**
 * Checked exception received when something fails during a call to a Resource 
 * Processor. A <code>ResourceProcessorException</code> always contains an error 
 * code (one of the constants specified in this class), and may optionally contain 
 * the textual description of the error condition and a nested cause exception.
 */
public class ResourceProcessorException extends Exception {

	/**
	 * The {@link ResourceProcessor#dropped(String)} method was called on a
	 * Resource Processor but the Resource Processor doesn't manage (doesn't know) 
	 * this resource.<p>
	 * 
	 * Only the {@link ResourceProcessor#dropped(String)} method is allowed to throw 
	 * exception with this error code.
	 */
	public static final int	CODE_NO_SUCH_RESOURCE			= 459;

	/**
	 * An artifact of any resource already exists.<p>
	 * 
	 * Only the {@link ResourceProcessor#process(String, InputStream)} method 
	 * is allowed to throw exception with this error code.  
	 */
	public static final int	CODE_RESOURCE_SHARING_VIOLATION	= 461;

	/**
	 * Other error condition.<p>
	 * 
	 * All Resource Processor methods which throw <code>ResourceProcessorException</code> 
	 * is allowed throw an exception with this erro code if the error condition cannot be 
	 * categorized. 
	 */
	public static final int	CODE_OTHER_ERROR				= 463;

	/**
	 * Resource Processors are allowed to raise an exception with this error code 
	 * to indicate that the processor is not able to commit the operations it made 
	 * since the last call of {@link ResourceProcessor#begin(DeploymentSession)} method.<p>
	 * 
	 * Only the {@link ResourceProcessor#prepare()} method is allowed to throw exception 
	 * with this error code.  
	 */
	public static final int	CODE_PREPARE					= 1;

	private int				code;
	private String			message;
	private Throwable		cause;

	/**
	 * Create an instance of the exception.
	 * 
	 * @param code The error code of the failure. Code should be one of the
	 *        predefined integer values (<code>CODE_X</code>).
	 * @param message Message associated with the exception
	 * @param cause the originating exception
	 */
	public ResourceProcessorException(int code, String message, Throwable cause) {
		this.code = code;
		this.message = message;
		this.cause = cause;
	}

	/**
	 * Create an instance of the exception. Cause exception is implicitly set to
	 * null.
	 * 
	 * @param code The error code of the failure. Code should be one of the
	 *        predefined integer values (<code>CODE_X</code>).
	 * @param message Message associated with the exception
	 */
	public ResourceProcessorException(int code, String message) {
		this(code, message, null);
	}

	/**
	 * Create an instance of the exception. Cause exception and message are
	 * implicitly set to null.
	 * 
	 * @param code The error code of the failure. Code should be one of the
	 *        predefined integer values (<code>CODE_X</code>).
	 */
	public ResourceProcessorException(int code) {
		this(code, null, null);
	}

	/**
	 * @return Returns the cause.
	 */
	public Throwable getCause() {
		return cause;
	}

	/**
	 * @return Returns the code.
	 */
	public int getCode() {
		return code;
	}

	/**
	 * @return Returns the message.
	 */
	public String getMessage() {
	    return message;
	}
}
